package modeselection;

import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.EnumSet;

import lejos.hardware.Button;
import lejos.hardware.lcd.LCD;

public class ModeSelector<C extends Enum<C>, M extends Enum<M>> implements Runnable {
	private Class<C> conditionClass;
	private M current;
	private EnumSet<C> conditions;
	private ArrayList<Flagger<C>> flaggers = new ArrayList<>();
	private ArrayList<SensorFlagger<C>> sensors = new ArrayList<>();
	private EnumMap<M,Runnable> actions;
	private EnumMap<M,Transitions<C,M>> modeTransitions;
	private Logger logger = Logger.EV3Log;
	
	public ModeSelector(Class<C> conditionClass, Class<M> modeClass, M start) {
		this.conditionClass = conditionClass;
		this.current = start;
		conditions = EnumSet.noneOf(conditionClass);
		actions = new EnumMap<>(modeClass);
		modeTransitions = new EnumMap<>(modeClass);
	}
	
	public ModeSelector<C,M> flagger(Flagger<C> flagger) {
		flaggers.add(flagger);
		return this;
	}
	
	public ModeSelector<C,M> sensor(SensorFlagger<C> flagger) {
		sensors.add(flagger);
		return flagger(flagger);
	}
	
	public ModeSelector<C,M> mode(M mode, Transitions<C,M> transitions, Runnable action) {
		actions.put(mode, action);
		modeTransitions.put(mode, transitions);
		return this;
	}
	
	public void control() throws IOException {
		actions.get(current).run();
		while (!Button.ESCAPE.isDown()) {
			run();
			show();
			log();
		}
		
		for (SensorFlagger<C> sensor: sensors) {
			sensor.close();
		}
	}

	@Override
	public void run() {
		conditions = EnumSet.noneOf(conditionClass);
		for (Flagger<C> sensor: flaggers) {
			sensor.update(conditions);
		}
		M prev = current;
		current = modeTransitions.get(current).getMode(current, conditions);
		if (prev != current) {
			actions.get(current).run();
		}
	}
	
	public void show() {
		LCD.clear();
		LCD.drawString(current.name(), 0, 0);
		int row = 1;
		for (C cond: conditions) {
			LCD.drawString(cond.name(), 0, row++);
		}
	}
	
	public void log() {
		logger.format("current: %s", current.name());
		logger.log("conditions:");
		for (C cond: conditions) {
			logger.log(cond.name());
		}
		for (Flagger<C> sensor: flaggers) {
			sensor.log(logger);
		}
	}
}
