package modeselection;

import java.io.IOException;
import java.util.EnumMap;
import java.util.EnumSet;

import lejos.hardware.Button;
import lejos.hardware.lcd.LCD;

public class ModeSelector<C extends Enum<C>, M extends Enum<M>> implements Runnable {
	private M current;
	private EnumSet<C> conditions;
	private StateClassifier<C> sensors;
	private EnumMap<M,Runnable> actions;
	private EnumMap<M,Transitions<C,M>> modeTransitions;
	private Logger logger = Logger.EV3Log;
	
	public ModeSelector(Class<C> conditionClass, Class<M> modeClass, M start) {
		this.current = start;
		conditions = EnumSet.noneOf(conditionClass);
		actions = new EnumMap<>(modeClass);
		modeTransitions = new EnumMap<>(modeClass);
		sensors = new StateClassifier<>(conditionClass);
	}
	
	public ModeSelector<C,M> flagger(Flagger<C> flagger) {
		sensors.add(flagger);
		return this;
	}
	
	public ModeSelector<C,M> sensor(SensorFlagger<C> flagger) {
		return flagger(flagger);
	}
	
	public ModeSelector<C,M> mode(M mode, Transitions<C,M> transitions, Runnable action) {
		actions.put(mode, action);
		modeTransitions.put(mode, transitions);
		return this;
	}
	
	public void control() throws IOException {
		actions.get(current).run();
		while (Button.ESCAPE.isUp()) {
			run();
			show();
			log();
		}
		sensors.close();
	}

	@Override
	public void run() {
		conditions = sensors.getCurrentState();
		M prev = current;
		current = modeTransitions.get(current).getMode(current, conditions);
		if (prev != current) {
			actions.get(current).run();
		}
	}
	
	public void show() {
		LCD.clear();
		LCD.drawString(current.name(), 0, 0);
		StateClassifier.showState(conditions, 1);
	}
	
	public void log() {
		logger.format("current: %s", current.name());
		sensors.logState(logger, conditions);
	}
}
