// Notes:
// I commented out the "Grammar" menu item.  
// It doesn't seem to work right.  I should revisit them at a later time.
// -GJF

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

import java.io.*;
import java.lang.reflect.Constructor;
import java.net.URL;
import java.net.URLClassLoader;

import javax.swing.border.EtchedBorder;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import squirrel.Grammar;
import squirrel.HasGrammar;
import squirrel.Parser;
import squirrel.Recognizer;
import squirrel.ResultTable;
import squirrel.Standard;
import squirrel.SymbolInfo;
import squirrel.Tree;

public class Gui extends JFrame implements DocumentListener, ActionListener, ChangeListener, CaretListener, MouseListener{
	
	private JPanel tableElms, textArea, toolPanel;
	private JTextArea inputField;
	private JMenuBar menubar;
	private JMenu menu;
	private JMenuItem item;
	private TreePanel treePanel;
	private ImageIcon b, f;
	private JButton back, forward;
	private JSlider slide;
	private ArrayList<TableEntry> table;
	private LinkedList<GrammarNode> visited;
	private TableEntry symb;
	private boolean entrySelected;

	private final int tableWidth;
	private final int tableHeight;
	private final int frameWidth;
	private final int frameHeight;
	private final int treePanelWidth;
	private final int treePanelHeight;
	private final int textAreaHeight;
	private final int textAreaWidth;

	private Parser p;
	private Grammar g;
	private TreeDisplay td;
	private GrammarNode rootGNode;

	private Dimension dims;
	
	public Gui() {
		this.dims = Toolkit.getDefaultToolkit().getScreenSize();
		this.frameHeight = dims.height;
		this.frameWidth = dims.width;
		this.tableHeight = dims.height;
		this.tableWidth = dims.width / 7;
		this.treePanelHeight = dims.height / 5;
		this.treePanelWidth = frameWidth - tableWidth;
		this.textAreaWidth = frameWidth - tableWidth;
		this.textAreaHeight = dims.height / 5;
		this.entrySelected = false;
		
		slide = new JSlider(JSlider.HORIZONTAL, 5, 30, 12);
		slide.setMinorTickSpacing(1);
		slide.setMajorTickSpacing(5);
		slide.setPaintTicks(true);
		slide.setPaintLabels(true);
		slide.addChangeListener(this);
		slide.setSnapToTicks(true);

		table = new ArrayList<TableEntry>();
		visited = new LinkedList<GrammarNode>();
		
		this.tableElms = new JPanel();
		this.tableElms.setPreferredSize(new Dimension(tableWidth, tableHeight));
		this.tableElms.setBackground(Color.lightGray);
		this.tableElms.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.RAISED));

		this.inputField = new JTextArea();
		this.inputField.setBorder(BorderFactory.createEtchedBorder(EtchedBorder.RAISED));
		this.inputField.setLineWrap(true);
		this.inputField.setPreferredSize(new Dimension(textAreaWidth, textAreaHeight / 2));
		this.inputField.getDocument().addDocumentListener(this);
		this.inputField.addCaretListener(this);
		
		this.textArea = new JPanel(new BorderLayout());
		this.textArea.setPreferredSize(new Dimension(textAreaWidth, textAreaHeight));
		this.textArea.add(new JScrollPane(inputField), BorderLayout.CENTER);
		
		treePanel = new TreePanel(this, dims);
		toolPanel = new JPanel(new BorderLayout());
		toolPanel.add(new JScrollPane(treePanel), BorderLayout.CENTER);
		toolPanel.add(textArea, BorderLayout.SOUTH);
		
		menubar = new JMenuBar();
		menu = new JMenu("Parsing");
		item = new JMenuItem("Set Grammar");
		item.addActionListener(this);
		menu.add(item);
		item = new JMenuItem("Load input from file");
		item.addActionListener(this);
		menu.add(item);
		menubar.add(menu);
		menu = new JMenu("View");
		item = new JMenuItem("Tokens Matched");
		item.addActionListener(this);
		menu.add(item);
		item = new JMenuItem("Text Matched");
		item.addActionListener(this);
		menu.add(item);
		menubar.add(menu);
		item = new JMenuItem("Entire Tree");
		item.addActionListener(this);
		menu.add(item);
		/*
		item = new JMenuItem("Grammar");
		item.addActionListener(this);
		menu.add(item);
		*/
		b = new ImageIcon("Back16.gif");
		back = new JButton(b);
		back.setEnabled(false);
		back.addActionListener(this);
		f = new ImageIcon("Forward16.gif");
		forward = new JButton(f);
		forward.setEnabled(false);
		forward.addActionListener(this);
		menubar.add(back);
		menubar.add(forward);
		menubar.add(slide);
		
		setPreferredSize(dims);
		setJMenuBar(menubar);
		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(new JScrollPane(tableElms), BorderLayout.WEST);
		getContentPane().add(toolPanel, BorderLayout.CENTER);
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setTitle("Squirrel Debugger");
	}
	
	/** pre: takes an unprocessed Grammar object
	 * 
	 * post: returns a tree structure representing the grammar's organization.  the root is the 
	 * start symbol, whose children are its possible matches, etc.
	 */
	public void processGrammar(GrammarNode cur) {
		Recognizer r = null; 
		r = g.ruleFor(cur.getName());
		if (r == null) {
			return;
		}
		for (int i = 0; i < r.numRows(); i++) {
			for (int j = 0; j < r.numSymbols(i); j++) {
				GrammarNode child = new GrammarNode(r.symbolAt(i, j));
				cur.addChild(child);
				child.setParent(cur);
				if (!child.getSymbolInfo().isTerminal())
				visited.offer(child);
			}
		}
		if (visited.size() != 0) {
			GrammarNode queued = visited.poll();
			processGrammar(queued);
		}
	}
	
	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();

		if (e.getSource() == back) {
			treePanel.back();
		}
		
		if (e.getSource() == forward) {
			treePanel.forward();
		}
		
		if (cmd.equals("Set Grammar")) {
			JFileChooser chooser = new JFileChooser();
			chooser.setDialogType(JFileChooser.OPEN_DIALOG);
			chooser.setFileFilter(new filter(".class"));
			int returnVal = chooser.showOpenDialog(this);			
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				File file = chooser.getSelectedFile();
				treePanel.clearStacks();
				loadGrammar(file);
			}
		}
		
		else if (cmd.equals("Load input from file")) {
			JFileChooser chooser = new JFileChooser();
			chooser.setDialogType(JFileChooser.OPEN_DIALOG);
			int returnVal = chooser.showOpenDialog(this);			
			if (returnVal == JFileChooser.APPROVE_OPTION) {
				File file = chooser.getSelectedFile();
				treePanel.clearStacks();
				loadInput(file);
			}
		}
		
		else if (cmd.equals("Tokens Matched")) {
			if (treePanel.getCurTree() != null)
			treePanel.getCurTree().tokenView(treePanel.getCurTree().getData());
		}
		
		else if (cmd.equals("Text Matched")) {
			if (treePanel.getCurTree() != null)
			treePanel.getCurTree().textView(treePanel.getCurTree().getData());
		}

		else if (cmd.equals("Entire Tree")) {
			if (treePanel.getCurTree() != null)
			treePanel.unfoldAll();
			treePanel.revalidate();
			treePanel.repaint();
		}
		else if (cmd.equals("Grammar")) {
			treePanel.getCurTree().drawGrammar(rootGNode, 0, 0);
		}
	}
	
	/**
	 * pre: grammar must be set by the user
	 * 
	 * post: the specified file is read, parsed, and displayed
	 */
	private void loadInput(File file) {
		StringBuilder s = new StringBuilder();
		try {
			Scanner in = new Scanner(file);
			while (in.hasNext()) {
				s.append(in.nextLine());
				s.append('\n');
			}
			inputField.setText(s.toString());
			
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * pre: none
	 * 
	 * post: through reflection, a grammar object is read in from a user file, set as 
	 * the current grammar, and parsed via the processGrammar method
	 */
	private void loadGrammar(File file) {
			try {
				URL[] urls = new URL[]{file.getParentFile().toURL()};
				String[] split = file.getName().split(".class");
				ClassLoader c = new URLClassLoader(urls);
				Class cl = c.loadClass(split[0]);
				Constructor con = cl.getConstructor(new Class[0]);
				HasGrammar hg = (HasGrammar) con.newInstance();
				g = hg.getGrammar();
				makeEntries();
				rootGNode = new GrammarNode(g.startSymbol());
				processGrammar(rootGNode);
			}
			catch (Exception e) {
				JOptionPane.showMessageDialog(null, "Bad Grammar File", "Error", JOptionPane.ERROR_MESSAGE);
				e.printStackTrace();
			}
	}

	
	/**
	 * pre: a valid grammar must be set and data present in the input field
	 * 
	 * post: input text is parsed, formatted, and displayed
	 */
	public void makeParser(String s) {
		if (g == null) {
			JOptionPane.showMessageDialog(null, "Grammar Not Set", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		p = new Parser(g, s);
		Tree spt = null;
		p.parse();
		spt = p.bestTree();
		td = new TreeDisplay(spt);
		td.makeDisplayTree(spt, 1, td.firstEntry());
		treePanel.setData(td.getData());
		treePanel.setFonts(slide.getValue());
		treePanel.setCurTree(treePanel);
		treePanel.fold();
		treePanel.repaint();
		treePanel.revalidate();
	}

	/**
	 * pre: a valid grammar must be set
	 *
	 * post: each symbol from the grammar is displayed as a clickable element on the screen
	 */
	private void makeEntries() {
		Iterator it = g.symbols().iterator();
		tableElms.removeAll();
		tableElms.setLayout(new GridLayout(g.symbols().size(), 1));
		TableEntry temp = null;
		while (it.hasNext()) {
			temp = new TableEntry((String)it.next(), tableElms.getGraphics());
			temp.addMouseListener(this);
			table.add(temp);
			tableElms.add(temp);
		}
		tableElms.setPreferredSize(new Dimension(tableElms.getWidth(), temp.getHeight() * g.symbols().size()));
		validate();
	}

	/**
	 * pre: a valid parser object must have been created and a grammar symbol must be selected
	 * 
	 * post: grabs the result table from the parser and displays the tree at the
	 * current input corresponding to the selected grammar symbol
	 */
	private void makeTable() {
		ResultTable table = null;
		try {
			table = p.results();
		} catch (NullPointerException e1) {
			
		}
		
		try {
			treePanel.push();
			Tree tree = table.getTree(symb.getName(), inputField.getCaretPosition());
			TreePanel curTree = new TreePanel(this, dims);
			TreeDisplay td = new TreeDisplay(tree);
			td.makeDisplayTree(tree, 1, td.firstEntry());
			curTree.setData(td.getData());
			treePanel.removeForwardElements();
			treePanel.setCurTree(curTree);
			treePanel.repaint();
		} catch (NullPointerException e) {
			treePanel.push();
			treePanel.removeAll();
			treePanel.add(new JLabel("No Tree at Current Input"));
			treePanel.repaint();
		}
	}

	public void changedUpdate(DocumentEvent e) {
	}

	public void insertUpdate(DocumentEvent e) {
		if (g == null) return;
		String input = inputField.getText();
		makeParser(input);
		treePanel.clearStacks();
		treePanel.repaint();
		treePanel.revalidate();
	}

	public void removeUpdate(DocumentEvent e) {
		if (g == null) return;
		String input = inputField.getText();
		makeParser(input);
		treePanel.clearStacks();
		treePanel.repaint();
		treePanel.revalidate();
	}

	public void setNavButtons() {
		if (treePanel.getBackStack().size() > 0) {
			back.setEnabled(true);
		}
		else {
			back.setEnabled(false);
		}

		if (treePanel.getForwardStack().size() > 0) {
			forward.setEnabled(true);
		}
		else {
			forward.setEnabled(false);
		}
	}

	public static void main(String[] args) throws IOException {
		Gui display = new Gui();
		display.setVisible(true);
		display.pack();
	}

	public void stateChanged(ChangeEvent e) {
		treePanel.getCurTree().setFonts(slide.getValue());
		treePanel.getCurTree().setVals(treePanel, treePanel.getCurTree().getWidths(treePanel.getGraphics()), treePanel.getGraphics());
		treePanel.getCurTree().setVals(treePanel, treePanel.getCurTree().getWidths(treePanel.getGraphics()), treePanel.getGraphics());
		treePanel.repaint();
	}

	public void caretUpdate(CaretEvent e) {
		if (p != null && entrySelected)
		makeTable();
	}

	public void mouseClicked(MouseEvent e) {
		// This handles clicking of grammar elements.
		if (entrySelected) {
			symb.setForeground(Color.black);
		}
		entrySelected = true;
		TableEntry entry = (TableEntry) e.getSource();
		entry.setForeground(Color.blue);
		symb = entry;
		makeTable();
	}

	public void mouseEntered(MouseEvent e) {
		
	}

	public void mouseExited(MouseEvent e) {
		
	}

	public void mousePressed(MouseEvent e) {
		
	}

	public void mouseReleased(MouseEvent e) {
		
	}
}
