import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.geom.Rectangle2D;
import java.util.*;

import javax.swing.*;

import squirrel.*;

public class TreePanel extends JPanel implements MouseListener{

	private HashMap<Integer, ArrayList<TreeNode>> data;
	private int fontSize;
	private double[] widths;
	private Dimension dims;
	private Gui gui;
	private Graphics gg;
	private TreePanel curTree;
	private Stack<TreePanel> backStack;
	private Stack<TreePanel> forwardStack;
	private LinkedList<GrammarNode> toDraw;

	public TreePanel(Gui gui, Dimension dims) {
		this.setLayout(new BorderLayout());
		this.dims = dims;
		this.fontSize = 15;
		this.setBackground(Color.lightGray);
		this.gui = gui;
		this.backStack = new Stack<TreePanel>();
		this.forwardStack = new Stack<TreePanel>();
		this.toDraw = new LinkedList<GrammarNode>();
	}

	public Gui getGui() {
		return gui;
	}

	public void setCurTree(TreePanel tp) {
		this.curTree = tp;
		curTree.widths = getWidths(gg);
		/*
		 * I call this method twice because I set the positions for all of the nodes, then
		 * check to see if some are drawn off the screen and adjust their positions.  I'm
		 * sure there's a better way to do this.
		 */
		curTree.setVals(this, curTree.widths, gg);
		curTree.setVals(this, curTree.widths, gg);
	}

	/**
	 * pre: a valid tree must be ready to be drawn on screen
	 *
	 * post: each node is drawn at the appropriate position and depth, with lines
	 * indicating relationships, and color indicating errors.  collapsed trees do not
	 * get drawn
	 */
	public void drawTree(Graphics g, Container con) {
		for (int i = 0; i < curTree.data.keySet().size(); i++) {
			ArrayList<TreeNode> nodes = curTree.data.get(i);
			for (int j = 0; j < nodes.size(); j++) {
				TreeNode curNode = nodes.get(j);
				if (curNode.isFolded()) {
					continue;
				}
				g.setColor(Color.black);
				curNode.setBounds(curNode.getX(), curNode.getY(), (int) curTree.getFontMetrics(curTree.getFont()).getStringBounds(curNode.getTree().name() + 1, g).getWidth(), curTree.getFontMetrics(curTree.getFont()).getHeight());
				if (curNode.getMouseListeners().length < 2) {
					curNode.addMouseListener(this);
				}
				if (curNode.getTree().numChildren() > 0) curNode.setBackground(Color.red);
				con.add(curNode);
				if (i != 0) {
					if (curNode.getTree().isError()) g.setColor(Color.red);
					g.drawLine(curNode.getTopCenter().x, curNode.getTopCenter().y, curNode.getParentTree().getBottomCenter().x, curNode.getParentTree().getBottomCenter().y);
				}
			}
		}
	}

	public void drawGrammar(GrammarNode gNode, int xStart, int yStart) {
		// Didn't get this far...
	}

	/**
	 * pre: a valid tree must exist
	 *
	 * post: the locations of every node in the tree are set according to their depth
	 * and spaced according to the number of elements at the current depth.
	 */
	public void setVals(Container con, double[] widths, Graphics g) {
		double x = 0;
		double y = 0;
		double[] bounds = getExtremeBounds(g);
		if (bounds[0] < 0) {
			x = -bounds[0];
            // This compensates for gradual degradation of the x
            // value that seems to occur when resizing.  
            x *= 1.1;
		}
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> temp = data.get(i);
			for (int j = 0; j < temp.size(); j++) {
				TreeNode blah = temp.get(j);
				if (blah.isFolded()) continue;
				if (blah.getParentTree() != null && j == 0) {
					x = blah.getParentTree().getX() - widths[i] / 2;
				}
				blah.setX((int)x);
				blah.setY((int)y);
				if (blah.getParentTree() != null)
				x += getFontMetrics(getFont()).getStringBounds(blah.getTree().name(), g).getWidth() + widths[i] / blah.getParentTree().getTree().numChildren() / 3;
			}
			y += getFontMetrics(getFont()).getHeight() * 5;
		}
	}

	/**
	 * pre: a valid tree must exist
	 *
	 * post: calculates the width of each level of the tree, for use in determining node
	 * positions
	 */
	public double[] getWidths(Graphics g) {
		double[] widths = new double[data.keySet().size()];
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> nodes = data.get(i);
			int rowWidth = 0;
			for (int j = 0; j < nodes.size(); j++) {
				if (!nodes.get(j).isFolded())
				rowWidth += getFontMetrics(getFont()).getStringBounds(nodes.get(j).getTree().name(), g).getWidth();
			}
			widths[i] = rowWidth;
		}
		return widths;
	}

	public void paintComponent(Graphics g) {
		this.gg = g;
		super.paintComponent(g);
		gui.setNavButtons();
		toDraw.clear();
		this.removeAll();
		if (curTree != null && curTree.data != null) {
			drawTree(g, this);
		}
	}

	/**
	 * pre: a valid tree must exist
	 *
	 * post: calculates the positions of the leftmost and rightmost nodes, for use in
	 * resizing the window and setting scrollbars
	 */
	public double[] getExtremeBounds(Graphics g) {
		double[] bounds = new double[2];
		bounds[0] = Integer.MAX_VALUE;
		bounds[1] = 0;
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> temp = data.get(i);
			for (int j = 0; j < temp.size(); j++) {
				TreeNode blah = temp.get(j);
				if (blah.isFolded()){
					continue;
				}
				if (blah.getX() < bounds[0]) {
					bounds[0] = blah.getX();
				}
				if (blah.getX() > bounds[1]) {
					bounds [1] = blah.getX() + getFontMetrics(getFont()).getStringBounds(blah.getTree().name(), g).getWidth();
				}
			}
		}
		return bounds;
	}

	public Dimension getPreferredSize() {
		if (widths == null) return new Dimension(dims.width - dims.width / 7, dims.height - dims.height / 5);;
		double[] bounds = getExtremeBounds(gg);
		return new Dimension((int) (bounds[1]), widths.length * curTree.getFontMetrics(curTree.getFont()).getHeight() * 5);
	}

	public void setData(HashMap<Integer, ArrayList<TreeNode>> data) {
		this.data = data;
	}

	public TreePanel getCurTree() {
		return curTree;
	}

	public HashMap<Integer, ArrayList<TreeNode>> getData() {
		return data;
	}


	/**
	 * pre: a mouseEvent is fired on a treeNode
	 *
	 * post: the clicked subtree is displayed, and the old tree pushed onto
	 * the stack
	 */
	public void mouseClicked(MouseEvent e) {
		TreeNode clicked = (TreeNode) e.getSource();
		if (e.getButton() == MouseEvent.BUTTON1) {
			if (clicked.getTree().numChildren() > 0) {
				backStack.push(curTree);
				curTree = new TreePanel(gui, dims);
				TreeDisplay td = new TreeDisplay(clicked.getTree());
				td.makeDisplayTree(clicked.getTree(), 1, td.firstEntry());
				curTree.setData(td.getData());
				forwardStack.removeAllElements();
				setCurTree(curTree);
			}
		}
		if (e.getButton() == MouseEvent.BUTTON3) {
			clicked.toggleText();
		}
		if(e.getButton() == MouseEvent.BUTTON2) {
			clicked.setFoldedRoot(!clicked.isFoldedRoot());
			curTree.fold();
		}
		revalidate();
		repaint();
	}

	/**
	 * pre: the user has middle-clicked on a node
	 *
	 * post: sets the clicked node and all of its children to folded
	 */
	public void fold() {
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> nodes = data.get(i);
			for (int j = 0; j < nodes.size(); j++) {
				TreeNode curNode = nodes.get(j);
				if (curNode.getParentTree() != null && (curNode.getParentTree().isFolded() || curNode.getParentTree().isFoldedRoot())) {
					curNode.setFolded(true);
				}
				else if (curNode.getParentTree() != null && (!curNode.getParentTree().isFolded() && !curNode.getParentTree().isFoldedRoot())) {
					curNode.setFolded(false);
				}
			}
		}
	}

	public void mouseEntered(MouseEvent e) {
	}

	public void mouseExited(MouseEvent e) {
	}

	public void mousePressed(MouseEvent e) {
	}

	public void mouseReleased(MouseEvent e) {
	}

	public Stack<TreePanel> getBackStack() {
		return backStack;
	}

	public Stack<TreePanel> getForwardStack() {
		return forwardStack;
	}

	public void setFonts(int k) {
		fontSize = k;
		setFont(new Font("Ariel", Font.PLAIN, k));
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> temp = data.get(i);
			for (int j = 0; j < temp.size(); j++) {
				temp.get(j).setFont(new Font("Ariel", Font.PLAIN, k));
			}
		}
	}


	public void tokenView(HashMap<Integer, ArrayList<TreeNode>> data) {
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> temp = data.get(i);
			for (int j = 0; j < temp.size(); j++) {
				if (temp.get(j).getTree().isError()) {
					temp.get(j).setText("Error");
				}
				else {
					temp.get(j).setText(temp.get(j).getTree().name());
					temp.get(j).setToolTipText(temp.get(j).getTree().name());
				}
			}
		}
	}


	public void textView(HashMap<Integer, ArrayList<TreeNode>> data) {
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> temp = data.get(i);
			for (int j = 0; j < temp.size(); j++) {
				if (temp.get(j).getTree().isError()) {
					temp.get(j).setText(temp.get(j).getTree().errorMessage());
				}
				else {
					temp.get(j).setText(temp.get(j).getTree().toString());
					temp.get(j).setToolTipText(temp.get(j).getTree().toString());
				}
			}
		}
	}

	public void back() {
		try {
			TreePanel temp = curTree;
			curTree = backStack.pop();
			forwardStack.push(temp);
			repaint();
		}
		catch (EmptyStackException e) {

		}
	}

	public void forward() {
		try {
			TreePanel temp = curTree;
			curTree = forwardStack.pop();
			backStack.push(temp);
			repaint();
		}
		catch (EmptyStackException e) {

		}
	}

	public void push() {
		backStack.push(curTree);
	}

	public void removeForwardElements() {
		forwardStack.removeAllElements();
	}

	public void pop() {
		curTree = forwardStack.pop();
	}

	public TreeNode firstEntry() {
		return data.get(0).get(0);
	}

	public void clearStacks() {
		if (backStack != null) backStack.removeAllElements();
		if (forwardStack != null) forwardStack.removeAllElements();
	}

	public void unfoldAll() {
		for (int i = 0; i < data.keySet().size(); i++) {
			ArrayList<TreeNode> temp = data.get(i);
			for (int j = 0; j < temp.size(); j++) {
				temp.get(j).setFolded(false);
				temp.get(j).setFoldedRoot(false);
			}
		}
	}
}
